import 'package:flutter/cupertino.dart';
import 'package:flutter/material.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:zcart_delivery/config/config.dart';
import 'package:zcart_delivery/helper/constants.dart';
import 'package:zcart_delivery/models/user_model.dart';
import 'package:zcart_delivery/providers/auth_provider.dart';
import 'package:zcart_delivery/providers/order_provider.dart';
import 'package:zcart_delivery/translations/locale_keys.g.dart';
import 'package:zcart_delivery/views/account/change_password_page.dart';
import 'package:zcart_delivery/views/account/profile_edit_page.dart';
import 'package:zcart_delivery/views/custom/update_language.dart';
import 'package:zcart_delivery/views/custom/version_info.dart';
import 'package:easy_localization/easy_localization.dart';

class AccountPage extends StatelessWidget {
  const AccountPage({super.key});

  @override
  Widget build(BuildContext context) {
    return DefaultTabController(
      length: 3,
      initialIndex: 0,
      child: Scaffold(
        endDrawer: _drawer(context),
        appBar: AppBar(title: Text(LocaleKeys.account.tr())),
        body: Consumer(
          builder: (context, ref, child) {
            final userRef = ref.watch(userProvider);
            return userRef.when(
              data: (data) {
                return data != null
                    ? ProfileSection(user: data)
                    : Center(
                        child: Text(LocaleKeys.something_went_wrong.tr()),
                      );
              },
              error: (error, stackTrace) {
                return Center(
                  child: Text(LocaleKeys.something_went_wrong.tr()),
                );
              },
              loading: () {
                return const Center(child: CircularProgressIndicator());
              },
            );
          },
        ),
      ),
    );
  }

  Widget _drawer(BuildContext context) {
    return Consumer(
      builder: (context, ref, child) {
        final userRef = ref.watch(userProvider);

        return Drawer(
          child: SafeArea(
            child: Column(
              children: [
                DrawerHeader(
                  child: Center(
                      child: userRef.when(
                    data: (data) {
                      return Column(
                        mainAxisAlignment: MainAxisAlignment.center,
                        children: [
                          ClipOval(
                            child: data?.avatar != null &&
                                    data!.avatar!.isNotEmpty
                                ? Image.network(
                                    data.avatar!,
                                    width: 60,
                                    height: 60,
                                    fit: BoxFit.cover,
                                    errorBuilder: (context, error, stackTrace) {
                                      return CircleAvatar(
                                        backgroundColor: MyConfig.primaryColor,
                                        radius: 30,
                                        child: const Icon(
                                          Icons.person,
                                          color: Colors.white,
                                          size: 30,
                                        ),
                                      );
                                    },
                                  )
                                : CircleAvatar(
                                    backgroundColor: MyConfig.primaryColor,
                                    radius: 30,
                                    child: const Icon(
                                      Icons.person,
                                      color: Colors.white,
                                      size: 30,
                                    ),
                                  ),
                          ),
                          const SizedBox(height: 8),
                          Text(data != null
                              ? (data.niceName ?? LocaleKeys.not_available.tr())
                              : LocaleKeys.not_available.tr()),
                          Text(data != null
                              ? data.email
                              : LocaleKeys.not_available.tr()),
                        ],
                      );
                    },
                    error: (error, stackTrace) {
                      return const SizedBox();
                    },
                    loading: () {
                      return const SizedBox();
                    },
                  )),
                ),
                Expanded(
                  child: ListView(
                    children: [
                      ListTile(
                        leading: const Icon(CupertinoIcons.lock),
                        title: Text(LocaleKeys.security.tr()),
                        onTap: () {
                          Navigator.push(
                            context,
                            MaterialPageRoute(
                              builder: (context) => const ChangePasswordPage(),
                            ),
                          );
                        },
                      ),
                      const Divider(height: 0, color: Colors.black12),
                      ListTile(
                        title: Text(
                          LocaleKeys.language.tr(),
                        ),
                        leading: const Icon(Icons.translate),
                        onTap: () {
                          updateLanguage(context);
                        },
                      ),
                      const Divider(height: 0, color: Colors.black12),
                      ListTile(
                        leading: const Icon(Icons.exit_to_app),
                        title: Text(LocaleKeys.logout.tr()),
                        onTap: () async {
                          showDialog(
                            context: context,
                            builder: (context) {
                              return AlertDialog(
                                title:
                                    Text(LocaleKeys.log_out_confirmation.tr()),
                                content: Text(LocaleKeys.log_out_content.tr()),
                                actions: [
                                  TextButton(
                                    child: Text(LocaleKeys.cancel.tr()),
                                    onPressed: () {
                                      Navigator.of(context).pop();
                                    },
                                  ),
                                  TextButton(
                                    child: Text(LocaleKeys.yes.tr()),
                                    onPressed: () async {
                                      Navigator.of(context).pop();
                                      final result =
                                          await AuthController.logout();
                                      if (result) {
                                        ref.invalidate(ordersProvider);
                                      }
                                    },
                                  ),
                                ],
                              );
                            },
                          );
                        },
                      ),
                      const Divider(height: 0, color: Colors.black12),
                    ],
                  ),
                ),
                const ShowVersionInfo(),
              ],
            ),
          ),
        );
      },
    );
  }
}

class ProfileSection extends StatelessWidget {
  final UserModel user;
  const ProfileSection({super.key, required this.user});

  @override
  Widget build(BuildContext context) {
    return SingleChildScrollView(
      padding: const EdgeInsets.all(defaultPadding * 1.5),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.stretch,
        children: [
          Center(
            child: ClipOval(
              child: user.avatar != null && user.avatar!.isNotEmpty
                  ? Image.network(
                      user.avatar!,
                      width: 100,
                      height: 100,
                      fit: BoxFit.cover,
                      errorBuilder: (context, error, stackTrace) {
                        return CircleAvatar(
                          backgroundColor: MyConfig.primaryColor,
                          radius: 50,
                          child: const Icon(
                            Icons.person,
                            color: Colors.white,
                            size: 50,
                          ),
                        );
                      },
                    )
                  : CircleAvatar(
                      backgroundColor: MyConfig.primaryColor,
                      radius: 50,
                      child: const Icon(
                        Icons.person,
                        color: Colors.white,
                        size: 50,
                      ),
                    ),
            ),
          ),
          const SizedBox(height: defaultPadding),
          Text(
            user.niceName ?? ("${user.firstName} ${user.lastName}"),
            textAlign: TextAlign.center,
            style: const TextStyle(
              fontSize: 20,
              fontWeight: FontWeight.bold,
            ),
          ),
          Text(
            user.email,
            textAlign: TextAlign.center,
            style: const TextStyle(
              fontSize: 16,
            ),
          ),
          const SizedBox(height: defaultPadding / 2),
          // Rating
          Row(
            mainAxisSize: MainAxisSize.min,
            mainAxisAlignment: MainAxisAlignment.center,
            children: [
              ...List.generate(
                5,
                (index) => Icon(
                  Icons.star,
                  color: user.avgRating > index
                      ? Colors.amber
                      : Colors.grey.shade300,
                  size: 16,
                ),
              ),
              const SizedBox(width: defaultPadding / 2),
              Text(
                user.avgRating.toStringAsFixed(1),
                style: Theme.of(context).textTheme.bodySmall!.copyWith(
                      fontWeight: FontWeight.bold,
                    ),
              ),
            ],
          ),

          // Center(
          //   child: OutlinedButton(
          //     onPressed: () {},
          //     child: Text(
          //       "View all reviews",
          //       style: Theme.of(context).textTheme.bodySmall!.copyWith(
          //           color: MyConfig.primaryColor, fontWeight: FontWeight.bold),
          //     ),
          //   ),
          // ),
          const SizedBox(height: defaultPadding),
          Card(
            elevation: 0,
            child: ListTile(
              dense: true,
              title: Text(LocaleKeys.nick_name.tr(),
                  style: Theme.of(context).textTheme.bodySmall),
              trailing: Text(
                user.niceName ?? LocaleKeys.not_available.tr(),
                style: Theme.of(context)
                    .textTheme
                    .bodyMedium!
                    .copyWith(fontWeight: FontWeight.bold),
              ),
            ),
          ),
          // First Name
          Card(
            elevation: 0,
            child: ListTile(
              dense: true,
              title: Text(LocaleKeys.first_name.tr(),
                  style: Theme.of(context).textTheme.bodySmall),
              trailing: Text(
                user.firstName,
                style: Theme.of(context)
                    .textTheme
                    .bodyMedium!
                    .copyWith(fontWeight: FontWeight.bold),
              ),
            ),
          ),
          // Last Name
          Card(
            elevation: 0,
            child: ListTile(
              dense: true,
              title: Text(LocaleKeys.last_name.tr(),
                  style: Theme.of(context).textTheme.bodySmall),
              trailing: Text(
                user.lastName,
                style: Theme.of(context)
                    .textTheme
                    .bodyMedium!
                    .copyWith(fontWeight: FontWeight.bold),
              ),
            ),
          ),
          // Email
          Card(
            elevation: 0,
            child: ListTile(
              dense: true,
              title: Text(LocaleKeys.email.tr(),
                  style: Theme.of(context).textTheme.bodySmall),
              trailing: Text(
                user.email,
                style: Theme.of(context)
                    .textTheme
                    .bodyMedium!
                    .copyWith(fontWeight: FontWeight.bold),
              ),
            ),
          ),
          // Phone Number
          Card(
            elevation: 0,
            child: ListTile(
              dense: true,
              title: Text(LocaleKeys.phone.tr(),
                  style: Theme.of(context).textTheme.bodySmall),
              trailing: Text(
                user.phoneNumber ?? LocaleKeys.not_available.tr(),
                style: Theme.of(context)
                    .textTheme
                    .bodyMedium!
                    .copyWith(fontWeight: FontWeight.bold),
              ),
            ),
          ),
          // Sex
          Card(
            elevation: 0,
            child: ListTile(
              dense: true,
              title: Text(LocaleKeys.sex.tr(),
                  style: Theme.of(context).textTheme.bodySmall),
              trailing: Text(
                user.sex ?? LocaleKeys.not_available.tr(),
                style: Theme.of(context)
                    .textTheme
                    .bodyMedium!
                    .copyWith(fontWeight: FontWeight.bold),
              ),
            ),
          ),
          const SizedBox(height: defaultPadding),
          ElevatedButton.icon(
            onPressed: () {
              Navigator.push(
                context,
                MaterialPageRoute(
                  builder: (context) => ProfileEditPage(user: user),
                ),
              );
            },
            label: Text(LocaleKeys.edit.tr()),
            icon: const Icon(Icons.edit),
          ),
        ],
      ),
    );
  }
}
